package utils;
use strict;
use warnings;
use utf8;
use Carp qw(croak);

use LWP::UserAgent;
use JSON;
my $ua = LWP::UserAgent->new;
my $json = JSON->new->utf8->indent;

sub json_api {
    my $url = shift;
    
    my $res = $ua->get(
        "https://api.github.com$url"
    );
    
    $res->is_success or croak $res->status_line;
    
    return $json->decode($res->content);
}


package GitHubBackup;
use base qw(Class::Accessor::Fast);

use strict;
use warnings;
use utf8;
use Carp qw(croak);

__PACKAGE__->mk_accessors( qw(
    directory
    account
    repository
));


# both hash and hashref are acceptable
sub new {
    my $class = shift;
    
    my $args = (ref $_[0] eq 'HASH') ? $_[0] : {@_};
    if (! $args->{directory}) {
        $args->{directory} = ".";
    }
    
    return $class->SUPER::new($args);
}

sub repos {
    my $self = shift;
    return $self->{repos} if ($self->{repos});
    
    my $account = $self->account or croak "account is not set";
    if (my $repository = $self->repository) {
        $self->{repos} = [
            GitHubBackup::Repository->new({
                directory => sub {$self->directory},
                full_name => "$account/$repository",
            })
        ];
        
        return $self->{repos};
    }
    
    my $page = 1;
    my @repos;
    while (1) {
        my $result = utils::json_api("/users/$account/repos?per_page=100&page=$page");
        if (ref($result) eq 'ARRAY' && scalar @$result > 0) {
            push @repos, @$result;
            $page++;
            
            next;
        }
        last;
    }
    
    foreach my $repos (@repos) {
        push @{$self->{repos}},
            GitHubBackup::Repository->new({
                directory => sub {$self->directory},
                full_name => $repos->{full_name},
                clone_url => $repos->{clone_url},
            })
        ;
    }
    
    return $self->{repos};
}


package GitHubBackup::Repository;
use base qw(Class::Accessor::Fast);

use strict;
use warnings;
use utf8;
use Carp qw(croak);
use Git::Repository;
use File::chdir;


sub new {
    my $class = shift;
    my $args  = shift;
    
    if (! exists $args->{clone_url}) {
        my $result = utils::json_api('/repos/' . $args->{full_name});
        $args->{clone_url} = $result->{clone_url};
    }
    
    return $class->SUPER::new($args);
}

sub directory {
    my $self = shift;
    
    return $self->{directory}->();
}

sub clone_git {
    my $self = shift;
    
    my $dir = $self->directory .'/'. $self->{full_name};
    if (-d "$dir") {
        local $CWD = $dir;
        print "fetch ", $self->{full_name}, "\n";
        Git::Repository->run(fetch => '--all');
        return $self;
    }
    
    print "clone ", $self->{full_name}, "\n";
    Git::Repository->run(clone => '--mirror' => $self->{clone_url} => $dir);
    return $self;
}

sub set_forks {
    my $self = shift;
}

sub clone_wiki {
    my $self = shift;
}

sub save_issues {
    my $self = shift;
}



1;
__END__
